//
//  TruckRoutePlanController.m
//  NavKitDemo
//
//  Created by mol on 2023/3/1.
//

#import "TruckRoutePlanController.h"
#import <TencentLBS/TencentLBS.h>

@interface TruckRoutePlanController () <TencentLBSLocationManagerDelegate>

// 定位管理类
@property (nonatomic, strong) TencentLBSLocationManager *locationManager;
// 是否获得了定位点
@property (nonatomic, assign) BOOL hasFirstLocation;
// 导航前记录最近的50个定位点，导航算路时做为前序点参数传入，以提高起点准确性
@property (nonatomic, strong) NSMutableArray<TencentLBSLocation *> *locations;

@end

@implementation TruckRoutePlanController

#pragma mark - lifecycle

- (void)dealloc {
    // 记得停止定位
    [self.locationManager stopUpdatingLocation];
    [self.locationManager stopUpdatingHeading];
}

- (void)viewDidLoad {
    [super viewDidLoad];
    
    self.locations = [NSMutableArray array];
    
    // 开启定位
    [self startLocationManager];
    
    [self setupBottomBar];    
}

- (void)viewWillAppear:(BOOL)animated {
    [super viewWillAppear:animated];
    
    [self.navigationController setToolbarHidden:NO animated:animated];
}

- (void)viewWillDisappear:(BOOL)animated {
    [super viewWillDisappear:animated];
    
    [self.navigationController setToolbarHidden:YES animated:animated];
}

#pragma mark - private

- (void)setupBottomBar {
    
    UIBarButtonItem *flexible = [[UIBarButtonItem alloc] initWithBarButtonSystemItem:UIBarButtonSystemItemFlexibleSpace
                                                                              target:nil
                                                                              action:nil];
    
    UIBarButtonItem *planRoutesItem = [[UIBarButtonItem alloc] initWithTitle:@"1.路径规划"
                                                                       style:UIBarButtonItemStyleDone
                                                                       target:self
                                                                       action:@selector(planRoutes)];
    UIBarButtonItem *startNavItem = [[UIBarButtonItem alloc] initWithTitle:@"2.开启导航"
                                                                        style:UIBarButtonItemStyleDone
                                                                       target:self
                                                                       action:@selector(startNav)];
    self.toolbarItems = @[flexible, planRoutesItem, flexible, startNavItem, flexible];
    self.navigationController.toolbar.backgroundColor = [UIColor colorWithWhite:1 alpha:0.8];
    
}

- (void)startLocationManager {
    
    self.locationManager = [[TencentLBSLocationManager alloc] init];
    // 设置定位的key，可以使用与地图SDK相同的key
    self.locationManager.apiKey = [QMapServices sharedServices].APIKey;
    self.locationManager.allowsBackgroundLocationUpdates = YES;
    // 监听定位信息
    self.locationManager.delegate = self;
 
    // 申请的定位权限，得和在info.list申请的权限对应才有效
    CLAuthorizationStatus authorizationStatus = [CLLocationManager authorizationStatus];
    if (authorizationStatus == kCLAuthorizationStatusNotDetermined) {
        [self.locationManager requestWhenInUseAuthorization];
    } else if (authorizationStatus == kCLAuthorizationStatusDenied) {
        [self showAlertWithError:[NSError errorWithDomain:@""
                                                     code:0
                                                 userInfo:@{NSLocalizedDescriptionKey : @"请去手机设置页面开启定位权限"}]];
    }
    
    // 开启定位
    [self.locationManager startUpdatingLocation];
    [self.locationManager startUpdatingHeading];
}

/// 初始路径规划
- (void)planRoutes {
    
    if (self.locations.count == 0) {
        [self showAlertWithError:[NSError errorWithDomain:@""
                                                     code:0
                                                 userInfo:@{NSLocalizedDescriptionKey : @"没有获得定位点，无法算路"}]];
        return;
    }
    
    TNKDriveRouteRequest *request = [[TNKDriveRouteRequest alloc] init];
    
    // 起点使用当前定位
    TNKSearchNavPoint *origin = [[TNKSearchNavPoint alloc] init];
    TNKSearchNavPoint *destination = [[TNKSearchNavPoint alloc] init];

    origin.coordinate = self.locations.lastObject.location.coordinate;
    
    // 交规1:
//    origin.coordinate = CLLocationCoordinate2DMake(39.951172, 116.425257);
//    destination.coordinate = CLLocationCoordinate2DMake(39.949241, 116.402161);
    
    // 交规2:
//    origin.coordinate = CLLocationCoordinate2DMake(39.916857,116.426539);
//    destination.coordinate = CLLocationCoordinate2DMake(39.915554, 116.429155);
    
    // 交规3: 禁止调头
//    origin.coordinate = CLLocationCoordinate2DMake(39.916857,116.426539);
//    destination.coordinate = CLLocationCoordinate2DMake(39.916416, 116.426727);
    
    // 交规4: 禁止调头
    origin.coordinate = CLLocationCoordinate2DMake(39.907522,116.277878);
    destination.coordinate = CLLocationCoordinate2DMake(39.907395, 116.276464);
    
    
    // 赋值
    request.origin = origin;
    request.destination = destination;
    
    // 算路策略
    TNKDriveRouteSearchOption *searchOption = [[TNKDriveRouteSearchOption alloc] init];
    searchOption.carNumber = @"桂C12345";
    // 货导参数
    TNKTruckSearchOption *truckSearchOption = [[TNKTruckSearchOption alloc] init];
    truckSearchOption.truckType = TNKTruckTypeMediumTruck;
    truckSearchOption.length = 8.0;
    truckSearchOption.width = 2.1;
    truckSearchOption.height = 4.2;
    truckSearchOption.weight = 25.0;
    truckSearchOption.load = 12.0;
    truckSearchOption.axisCount = 6;
    truckSearchOption.axisLoad = 1.0;
    truckSearchOption.plateColor = TNKTruckPlateColorBlue;
    
    searchOption.truckSearchOption = truckSearchOption;
    request.searchOption = searchOption;
    
    
    
    __weak typeof(self) weakself = self;

    // 初始路径规划
    [self.navManager searchRoutesWithRequest:request
                                  completion:^(TNKDriveRouteSearchResult * _Nullable searchResult,
                                               NSError * _Nullable error) {
       
        if (error != nil) {
            // 算路失败
            [weakself showAlertWithError:error];
            return;
        }
        
        // 清空地图上的元素
        [weakself clearRoutePlan];
        // 地图上绘制路线，并且让第一条路线高亮
        [weakself drawRoutePlan:searchResult selectedIndex:0];
    }];
}

/// 开启导航
- (void)startNav {
    // 清除导航前绘制的元素
    [self clearRoutePlan];
    
    // 展示默认导航控件
    self.navView.showUIElements = YES;
    // 使用第一条路线进行导航
    [self.navManager startNavWithIndex:0];
    
    // 隐藏toolbar
    [self.navigationController setToolbarHidden:YES animated:NO];
}



#pragma mark - TencentLBSLocationManagerDelegate
/**
 *  当定位发生错误时，会调用代理的此方法
 *
 *  @param manager 定位 TencentLBSLocationManager 类
 *  @param error 返回的错误，参考 TencentLBSLocationError
 */
- (void)tencentLBSLocationManager:(TencentLBSLocationManager *)manager
                 didFailWithError:(NSError *)error {
    
    NSLog(@"tencentLBSLocationManager:didFailWithError:%@", error);
}

/**
 *  连续定位回调函数
 *
 *  @param manager 定位 TencentLBSLocationManager 类
 *  @param location 定位结果
 */
- (void)tencentLBSLocationManager:(TencentLBSLocationManager *)manager
                didUpdateLocation:(TencentLBSLocation *)location {
//    NSLog(@"didUpdateLocation=%@", location);

    if (!self.hasFirstLocation) {
        // 获取到的第一个定位，移动地图至当前位置
        self.mapView.centerCoordinate = location.location.coordinate;
        self.mapView.zoomLevel = 15;
        self.hasFirstLocation = YES;
        
    }
        
    // 记录最新50个定位点
    [self.locations addObject:location];
    if (self.locations.count > 50) {
        [self.locations removeObjectAtIndex:0];
    }
    
}


@end
