//
//  LandscapeModeView.m
//  NavKitDemo
//
//  Created by mol on 2023/5/19.
//

#import "LandscapeModeView.h"
#import "DeviceUtil.h"

@interface CurrentSpeedView : UIView
@property (weak, nonatomic) IBOutlet UILabel *currentSpeedLabel;

@end


@implementation CurrentSpeedView

- (void)awakeFromNib {
    [super awakeFromNib];

    self.backgroundColor = [UIColor whiteColor];
    self.layer.cornerRadius = self.bounds.size.width / 2;
    self.layer.shadowColor = [UIColor lightGrayColor].CGColor;
    self.layer.shadowOffset = CGSizeMake(0, 1);
    self.layer.shadowRadius = 2;
    self.layer.shadowOpacity = 0.4;
}

@end

@interface NavTopPanelView : UIView

@property (weak, nonatomic) IBOutlet UIImageView *turnImageView;
@property (weak, nonatomic) IBOutlet UILabel *distanceLabel;
@property (weak, nonatomic) IBOutlet UILabel *distanceUnitLabel;
@property (weak, nonatomic) IBOutlet UILabel *actionLabel;
@property (weak, nonatomic) IBOutlet UILabel *roadNameLabel;

@property (strong, nonatomic) IBOutletCollection(NSLayoutConstraint) NSArray *portraitConstraints;

@property (strong, nonatomic) IBOutletCollection(NSLayoutConstraint) NSArray *landscapeConstraints;

                                                 
@end

@implementation NavTopPanelView


- (void)awakeFromNib {
    [super awakeFromNib];
    
    self.hidden = YES;
}

- (void)switch2Orientation:(UIInterfaceOrientation)deviceOrientation {
    
    if (UIInterfaceOrientationIsPortrait(deviceOrientation)) {
        // 切到竖屏
        [NSLayoutConstraint deactivateConstraints:self.landscapeConstraints];
        [NSLayoutConstraint activateConstraints:self.portraitConstraints];
        
    } else {
        // 切到横屏
        [NSLayoutConstraint deactivateConstraints:self.portraitConstraints];
        [NSLayoutConstraint activateConstraints:self.landscapeConstraints];

    }
}

// 更新导航信息
- (void)updateNavigationData:(TNKDriveNavigationData *)navigationData {
    
    if (self.hidden) {
        self.hidden = NO;
    }
    self.turnImageView.image = navigationData.intersectionImage;
    self.roadNameLabel.text = navigationData.nextRoadName;
    
    if (navigationData.nextDistanceLeft > 1000) {
        // 大于一公里，单位展示为公里
        self.distanceLabel.text = [NSString stringWithFormat:@"%.1f", navigationData.nextDistanceLeft / 1000.0];
        self.distanceUnitLabel.text = @"公里";
    } else if (navigationData.nextDistanceLeft > 10) {
        // 大于十米，单位展示为米
        self.distanceLabel.text = [NSString stringWithFormat:@"%d", navigationData.nextDistanceLeft];
        self.distanceUnitLabel.text = @"米";
    } else {
        // 展示为 "现在"
        self.distanceLabel.text = @"现在";
        self.distanceUnitLabel.text = @"";
    }
}

@end



@interface LandscapeModeView ()

@property (weak, nonatomic) IBOutlet NavTopPanelView *topPanelView;
@property (weak, nonatomic) IBOutlet CurrentSpeedView *currentSpeedView;
@property (weak, nonatomic) IBOutlet UIImageView *laneImageView;
@property (weak, nonatomic) IBOutlet TNKNaviTrafficBarWidget *trafficBarWidget;


@property (weak, nonatomic) IBOutlet UIButton *navModeSwitchBtn;
@property (weak, nonatomic) IBOutlet UIButton *landscapeModeSwitchBtn;
@property (weak, nonatomic) IBOutlet UIButton *continueBtn;

@property (strong, nonatomic) IBOutletCollection(NSLayoutConstraint) NSArray *portraitConstraints;

@property (strong, nonatomic) IBOutletCollection(NSLayoutConstraint) NSArray *landscapeConstraints;
@property (weak, nonatomic) IBOutlet NSLayoutConstraint *portaitCurrentSpeedViewTop;
@property (weak, nonatomic) IBOutlet NSLayoutConstraint *protaitLaneImageViewTop;
@property (weak, nonatomic) IBOutlet NSLayoutConstraint *protraitTopPanelTop;

@property (nonatomic, assign) UIInterfaceOrientation deviceOrientation;
@property (nonatomic, assign) TNKNavUIMode navUIMode;

// 是否正在展示车道线
@property (nonatomic, assign) BOOL showingLane;

@end

const CGFloat kTopPanelMargin = 10;
const CGFloat kTopPanelRatio = 7.0 / 2.0;
const CGFloat kLaneHeight = 50.0;

@implementation LandscapeModeView

+ (instancetype)landscapeModeView {
    
    LandscapeModeView * landscapeModeView = [[UINib nibWithNibName:@"LandscapeModeView" bundle:nil] instantiateWithOwner:nil options:nil].firstObject;
    return landscapeModeView;
}

- (instancetype)initWithFrame:(CGRect)frame {
    self = [super initWithFrame:frame];
    if (self) {
        [self setup];
    }
    
    return self;
}

- (void)awakeFromNib {
    [super awakeFromNib];
    
    [self setup];
}

- (void)setup {
    self.followed = YES;

}

- (UIView *)hitTest:(CGPoint)point withEvent:(UIEvent *)event
{
    UIView *hitView = [super hitTest:point withEvent:event];

    // If the hitView is THIS view, return nil and allow hitTest:withEvent: to
    // continue traversing the hierarchy to find the underlying view.
    if (hitView == self) {
        return nil;
    }
    // Else return the hitView (as it could be one of this view's buttons):
    return hitView;
}

#pragma mark - action

- (IBAction)switchNavMode:(id)sender {
    TNKNavUIMode navUIMode = TNKNavUIMode3DTowardsUp == self.navUIMode ? TNKNavUIModeRemainingOverview : TNKNavUIMode3DTowardsUp;
    
    if ([self.delegate respondsToSelector:@selector(landscapeModeView:didClickSwitchNavMode:)]) {
        [self.delegate landscapeModeView:self didClickSwitchNavMode:navUIMode];
    }
}

- (IBAction)switchLandscapeMode:(id)sender {
    
    UIInterfaceOrientation deviceOrientation = UIInterfaceOrientationIsPortrait(self.deviceOrientation) ? UIInterfaceOrientationLandscapeRight : UIInterfaceOrientationPortrait;
   
    if ([self.delegate respondsToSelector:@selector(landscapeModeView:didClickSwitchOrientation:)]) {
        [self.delegate landscapeModeView:self didClickSwitchOrientation:deviceOrientation];
    }
}

- (IBAction)contiueNav:(id)sender {
    
    if ([self.delegate respondsToSelector:@selector(landscapeModeViewContinueNav:)]) {
        [self.delegate landscapeModeViewContinueNav:self];
    }
}

#pragma mark - public

- (void)switch2Orientation:(UIInterfaceOrientation)deviceOrientation {
    
    [self.topPanelView switch2Orientation:deviceOrientation];

    if (UIInterfaceOrientationIsPortrait(deviceOrientation)) {
        // 切到竖屏
        [NSLayoutConstraint deactivateConstraints:self.landscapeConstraints];
        [NSLayoutConstraint activateConstraints:self.portraitConstraints];
        
        [self.landscapeModeSwitchBtn setTitle:@"横屏" forState:UIControlStateNormal];

        CGFloat topPanelHeight = ([UIScreen mainScreen].bounds.size.width - 2 * kTopPanelMargin) / kTopPanelRatio;
        CGFloat topPanelTop = [DeviceUtil TopNotchHeight];
        _externalEdgeInsets = UIEdgeInsetsMake(topPanelTop + topPanelHeight + kLaneHeight, 20, 24, 20);
        self.protraitTopPanelTop.constant = topPanelTop;

    } else {
        // 切到横屏
        [NSLayoutConstraint deactivateConstraints:self.portraitConstraints];
        [NSLayoutConstraint activateConstraints:self.landscapeConstraints];

        [self.landscapeModeSwitchBtn setTitle:@"竖屏" forState:UIControlStateNormal];
        
        _externalEdgeInsets = UIEdgeInsetsMake(80, 240, 40, 40);
    }
    
    self.deviceOrientation = deviceOrientation;
}

- (void)switch2NavMode:(TNKNavUIMode)navUIMode {
    self.navUIMode = navUIMode;
    
    if (navUIMode == TNKNavUIMode3DTowardsUp) {
        [self.navModeSwitchBtn setTitle:@"全览" forState:UIControlStateNormal];
    } else {
        [self.navModeSwitchBtn setTitle:@"退出全览" forState:UIControlStateNormal];

    }
}

// 更新导航信息
- (void)updateNavigationData:(TNKDriveNavigationData *)navigationData {
    // 更新当前车速
    self.currentSpeedView.currentSpeedLabel.text = navigationData.currentSpeed >= 0 ? [NSString stringWithFormat:@"%d", navigationData.currentSpeed] : @"--";
    
    [self.topPanelView updateNavigationData:navigationData];
}

- (NSArray<id<TNKDriveNavDelegate>> *)navDelegates {
    return @[self, self.trafficBarWidget];
}

- (void)setShowingEnlargedMap:(BOOL)showingEnlargedMap {
    if (showingEnlargedMap == _showingEnlargedMap) {
        return;
    }
    _showingEnlargedMap = showingEnlargedMap;
    
    self.trafficBarWidget.alpha = showingEnlargedMap ? 0.0 : 1.0;
    
    [self layoutNavWidgets];
}

- (void)setShowingLane:(BOOL)showingLane {
    if (showingLane == _showingLane) {
        return;
    }
    _showingLane = showingLane;
}

- (void)setFollowed:(BOOL)followed {
    if (followed == _followed) {
        return;
    }
    _followed = followed;
    
    self.continueBtn.hidden = followed;
}

#pragma mark - private

// 更新导航控件布局
- (void)layoutNavWidgets {
    CGFloat topPanelHeight = ([UIScreen mainScreen].bounds.size.width - 2 * kTopPanelMargin) / kTopPanelRatio;
    CGFloat topPanelTop = [DeviceUtil TopNotchHeight];
    
    if (_showingEnlargedMap) {
        
        CGRect enlargedMapRect = [self.delegate enlargedMapRectForLandscapeModeView:self];
        CGFloat protaitLaneImageViewTopConstant = CGRectGetMaxY(enlargedMapRect) - kLaneHeight - topPanelTop - topPanelHeight;
        if (protaitLaneImageViewTopConstant != self.protaitLaneImageViewTop.constant) {
            self.protaitLaneImageViewTop.constant = protaitLaneImageViewTopConstant;
        }
        
        if (UIInterfaceOrientationIsPortrait(self.deviceOrientation)) {
            self.topPanelView.alpha = 0.0;
        } else {
            self.topPanelView.alpha = 1.0;
        }
    } else {
        self.protaitLaneImageViewTop.constant = 10;
        self.topPanelView.alpha = 1.0;
    }
}

#pragma mark - TNKDriveNavDelegate

- (void)driveNavManager:(TNKDriveNavManager *)manager showGuidedLaneInfo:(TNKLaneInfo *)laneInfo {
    self.laneImageView.layer.masksToBounds = YES;
    self.laneImageView.hidden = NO;
    self.showingLane = YES;
}

- (void)driveNavManagerHideGuidedLaneInfo:(TNKDriveNavManager *)manager {
    self.laneImageView.image = nil;
    self.laneImageView.hidden = YES;
    self.showingLane = NO;
    
}

- (void)driveNavManagerDidStart:(TNKDriveNavManager *)manager {
    self.showingEnlargedMap = NO;
    self.showingLane = NO;
}

- (void)driveNavManagerDidStop:(TNKDriveNavManager *)manager {
    self.showingEnlargedMap = NO;
    self.showingLane = NO;
}

- (void)driveNavManager:(TNKDriveNavManager *)manager updateNavigationData:(TNKDriveNavigationData *)navigationData {
    [self updateNavigationData:navigationData];
}


@end
