//
//  DriveLandscapeModeController.m
//  NavKitDemo
//
//  Created by mol on 2023/5/19.
//

#import "DriveLandscapeModeController.h"
#import "LandscapeModeView.h"
#import "DeviceUtil.h"
#import <TencentNavKit/TencentNavKit.h>

@interface DriveLandscapeModeController () <LandscapeModeViewDelegate>

@property (nonatomic, strong) LandscapeModeView *landscapeModeView;

@end

@implementation DriveLandscapeModeController

- (void)dealloc {
    NSArray<id<TNKDriveNavDelegate>> *navDelegates = [self.landscapeModeView.navDelegates copy];
    for (id<TNKDriveNavDelegate> navDelegate in navDelegates) {
        [self.navManager registerNavDelegate:navDelegate];
    }
}

- (void)viewDidLoad {
    [super viewDidLoad];
    
    self.navView.navigationPanelEnabled = NO;
    self.navView.guidedLaneEnabled = NO;
    self.navView.showUIElements = NO;
        
    [self setupLandscapeModeView];
    
    // 发起算路
    [self planRoutes];
}

- (void)viewWillAppear:(BOOL)animated {
    [super viewWillAppear:animated];
    
    // 页面默认竖屏展示
    [self switch2Orientation:UIInterfaceOrientationPortrait];
}

- (void)viewWillDisappear:(BOOL)animated {
    [super viewWillDisappear:animated];
    
    // 离开页面切换回竖屏
    [DeviceUtil setCurrentOrientation:UIInterfaceOrientationPortrait];
}

#pragma mark - chnange orientation

- (void)switch2Orientation:(UIInterfaceOrientation)deviceOrientation {
    
    [DeviceUtil setCurrentOrientation:deviceOrientation];
        
    if (self.navView.navUIMode == TNKNavUIModeRemainingOverview) {
        [self switch2NavMode:TNKNavUIMode3DTowardsUp];
    }
}

- (void)switch2NavMode:(TNKNavUIMode)navUIMode {
    self.navView.navUIMode = navUIMode;
    [self.landscapeModeView switch2NavMode:navUIMode];
}

#pragma mark - private

- (void)setupLandscapeModeView {
    
    self.landscapeModeView = [LandscapeModeView landscapeModeView];
    [self.landscapeModeView switch2Orientation:UIInterfaceOrientationPortrait];
    [self.landscapeModeView switch2NavMode:TNKNavUIMode3DTowardsUp];
    self.landscapeModeView.delegate = self;
    self.landscapeModeView.frame = self.view.bounds;
    self.landscapeModeView.autoresizingMask = UIViewAutoresizingFlexibleWidth | UIViewAutoresizingFlexibleHeight;
    [self.view addSubview:self.landscapeModeView];
    
    NSArray<id<TNKDriveNavDelegate>> *navDelegates = [self.landscapeModeView.navDelegates copy];
    for (id<TNKDriveNavDelegate> navDelegate in navDelegates) {
        [self.navManager registerNavDelegate:navDelegate];
    }
    
}

/// 初始路径规划
- (void)planRoutes {
    
    __weak typeof(self) weakself = self;
    
    TNKDriveRouteRequest *request = [[TNKDriveRouteRequest alloc] init];
    
    // 起点信息
    TNKSearchNavPoint *origin = [[TNKSearchNavPoint alloc] init];
    origin.coordinate = CLLocationCoordinate2DMake(40.038591, 116.273023);
    
    // 终点信息. 可以使用关键词检索获得坐标、poi id、名称等信息。
    // 详情见: https://lbs.qq.com/service/webService/webServiceGuide/webServiceSuggestion
    TNKSearchNavPoint *destination = [[TNKSearchNavPoint alloc] init];
    destination.coordinate = CLLocationCoordinate2DMake(39.907773, 116.401273);
    destination.poiID = @"2199027907804";
    destination.title = @"天安门东[地铁站]";
    
    // 赋值
    request.origin = origin;
    request.destination = destination;
    
    // 算路策略
    TNKDriveRouteSearchOption *searchOption = [[TNKDriveRouteSearchOption alloc] init];
    // 导航场景. 1表示去接乘客, 2表示去送乘客. 默认值为0.
    searchOption.navScene = 1;
    request.searchOption = searchOption;

    // 初始路径规划
    [self.navManager searchRoutesWithRequest:request
                                  completion:^(TNKDriveRouteSearchResult * _Nullable searchResult,
                                               NSError * _Nullable error) {
       
        if (error != nil) {
            // 算路失败
            [weakself showAlertWithError:error];
            return;
        }
        
        // 算路成功，开始模拟导航
        [weakself startNav];
    }];
}

- (void)startNav {
    // 使用第一条路线进行模拟导航
    self.navManager.locationSimulator.enabled = YES;
    [self.navManager startNavWithIndex:0];
}

#pragma mark - LandscapeModeViewDelegate

- (void)landscapeModeView:(LandscapeModeView *)landscapeModeView
didClickSwitchOrientation:(UIInterfaceOrientation)deviceOrientation {
    
    [self switch2Orientation:deviceOrientation];
}

- (void)landscapeModeView:(LandscapeModeView *)landscapeModeView
    didClickSwitchNavMode:(TNKNavUIMode)navUIMode {
    
    [self switch2NavMode:navUIMode];
}

- (void)landscapeModeViewContinueNav:(LandscapeModeView *)landscapeModeView {
    self.navView.followed = YES;
}

- (CGRect)enlargedMapRectForLandscapeModeView:(LandscapeModeView *)landscapeModeView {
    return [self calcEnlargedMapRectWithViewSize:self.view.bounds.size];
}

- (CGRect)calcEnlargedMapRectWithViewSize:(CGSize)viewSize {
    if (UIInterfaceOrientationIsPortrait([DeviceUtil currentOrientation])) {
        // 竖屏
        CGFloat margin = 10;
        return CGRectMake(margin, [DeviceUtil TopNotchHeight], viewSize.width - 2 * margin, (viewSize.width - 2 * margin) / kTNKEnlargedIntersectionAspect + kTNKEnlargedIntersectionProgressHeight);
    } else {
        // 横屏
        return CGRectMake(viewSize.width - 200 - 10, 20, 200, 200 / kTNKEnlargedIntersectionAspect);
    }
}

#pragma mark - TNKDriveNavDelegate

- (void)navViewDidShowEnlargedMap:(TNKDriveNavView *)navView {
    self.landscapeModeView.showingEnlargedMap = YES;
}

- (void)navViewDidHideEnlargedMap:(TNKDriveNavView *)navView {
    self.landscapeModeView.showingEnlargedMap = NO;
}

- (void)navView:(TNKBaseNavView *)navView didChangeUIMode:(TNKNavUIMode)mode followed:(BOOL)followed {
    self.landscapeModeView.followed = followed;
}

#pragma mark -
- (void)viewWillTransitionToSize:(CGSize)size withTransitionCoordinator:(id<UIViewControllerTransitionCoordinator>)coordinator {
    [super viewWillTransitionToSize:size withTransitionCoordinator:coordinator];
    
    __weak typeof(self) weakself = self;
    [coordinator animateAlongsideTransition:nil completion:^(id<UIViewControllerTransitionCoordinatorContext>  _Nonnull context) {
        //
        [weakself rotated];
    }];
}

- (void)rotated {
    
    CGRect enlargedMapRect = [self calcEnlargedMapRectWithViewSize:self.view.bounds.size];
    [self.landscapeModeView switch2Orientation:[DeviceUtil currentOrientation]];

    if (UIInterfaceOrientationIsPortrait([DeviceUtil currentOrientation])) {
        // 竖屏
        self.navView.fix3DProportion = CGPointMake(0.5, 0.72);
        self.navView.fix2DProportion = CGPointMake(0.5, 0.6);

        TNKEnlargedMapConfig *enlargedMapConfig = self.navView.enlargedMapConfig;
        enlargedMapConfig.showEnlargedMapProgressView = YES;
        enlargedMapConfig.enlargedMapRect = enlargedMapRect;
        enlargedMapConfig.bottomLeftCornerRadius = 20;
        enlargedMapConfig.bottomRightCornerRadius = 20;

        [self.navView updateEnlargedMapConfig:enlargedMapConfig];

        self.navView.externalEdgeInsets = self.landscapeModeView.externalEdgeInsets;
    } else {
        // 横屏
        self.navView.fix3DProportion = CGPointMake(0.6, 0.76);
        self.navView.fix2DProportion = CGPointMake(0.6, 0.6);
        
        TNKEnlargedMapConfig *enlargedMapConfig = self.navView.enlargedMapConfig;
        enlargedMapConfig.enlargedMapRect = enlargedMapRect;
        enlargedMapConfig.showEnlargedMapProgressView = NO;
        enlargedMapConfig.bottomLeftCornerRadius = 0;
        enlargedMapConfig.bottomRightCornerRadius = 0;
        [self.navView updateEnlargedMapConfig:enlargedMapConfig];
        
        self.navView.externalEdgeInsets = self.landscapeModeView.externalEdgeInsets;
    }
    
}

@end
