//
//  DriveCustomTrafficBarController.m
//  NavKitDemo
//
//  Created by mol on 2023/2/9.
//

#import "DriveCustomTrafficBarController.h"
#import "DriveCustomTrafficBar.h"
#import "DeviceUtil.h"

#define CUSTOM_TRAFFIC_BAR 0

@interface DriveCustomTrafficBarController ()

#if CUSTOM_TRAFFIC_BAR
@property (nonatomic, strong) DriveCustomTrafficBar *trafficBar;
#else
@property (nonatomic, strong) TNKNaviTrafficBarWidget *trafficBar;
#endif

@end

@implementation DriveCustomTrafficBarController

- (void)viewDidLoad {
    [super viewDidLoad];
        
    // 发起算路
    [self planRoutes];
    
    // 自定义光柱图
    [self setupCustomTrafficBar];
}

#pragma mark - private

/// 初始化自定义面板
- (void)setupCustomTrafficBar {
    CGFloat rightMargin = 15;
    CGFloat trafficBarWidth = 20;
    CGFloat top = [DeviceUtil topPadding] + 120;
    
#if CUSTOM_TRAFFIC_BAR
    self.trafficBar = [[DriveCustomTrafficBar alloc] initWithFrame:CGRectMake(self.view.bounds.size.width - rightMargin - trafficBarWidth,
                                                                                    top,
                                                                                    trafficBarWidth,
                                                                                    self.view.bounds.size.height - top - 80)];
    self.trafficBar.hidden = YES;
    [self.view addSubview:self.trafficBar];
#else
    self.trafficBar = [[TNKNaviTrafficBarWidget alloc] initWithFrame:CGRectMake(self.view.bounds.size.width - rightMargin - trafficBarWidth,
                                                                                    top,
                                                                                    trafficBarWidth,
                                                                                    self.view.bounds.size.height - top - 80)];
    self.trafficBar.hidden = YES;
    [self.view addSubview:self.trafficBar];
    [self.navManager registerNavDelegate:self.trafficBar];
#endif
    
}

/// 初始路径规划
- (void)planRoutes {
    
    __weak typeof(self) weakself = self;
    
    TNKDriveRouteRequest *request = [[TNKDriveRouteRequest alloc] init];
    
    // 起点信息
    TNKSearchNavPoint *origin = [[TNKSearchNavPoint alloc] init];
    origin.coordinate = CLLocationCoordinate2DMake(40.038591, 116.273023);
    
    // 终点信息. 可以使用关键词检索获得坐标、poi id、名称等信息。
    // 详情见: https://lbs.qq.com/service/webService/webServiceGuide/webServiceSuggestion
    TNKSearchNavPoint *destination = [[TNKSearchNavPoint alloc] init];
    destination.coordinate = CLLocationCoordinate2DMake(39.907773, 116.401273);
    destination.poiID = @"2199027907804";
    destination.title = @"天安门东[地铁站]";
    
    // 赋值
    request.origin = origin;
    request.destination = destination;
    
    // 算路策略
    TNKDriveRouteSearchOption *searchOption = [[TNKDriveRouteSearchOption alloc] init];
    // 导航场景. 1表示去接乘客, 2表示去送乘客. 默认值为0.
    searchOption.navScene = 1;
    request.searchOption = searchOption;

    // 初始路径规划
    [self.navManager searchRoutesWithRequest:request
                                  completion:^(TNKDriveRouteSearchResult * _Nullable searchResult,
                                               NSError * _Nullable error) {
       
        if (error != nil) {
            // 算路失败
            [weakself showAlertWithError:error];
            return;
        }
        
        // 算路成功，开始模拟导航
        [weakself startNav];
    }];
}

- (void)startNav {
    // 使用第一条路线进行模拟导航
    self.navManager.locationSimulator.enabled = YES;
    [self.navManager startNavWithIndex:0];
}

#pragma mark - TNKDriveNavDelegate
- (void)driveNavManagerDidStart:(TNKDriveNavManager *)manager {
    // 开始导航， 取路况数据
    self.trafficBar.hidden = NO;
    
#if CUSTOM_TRAFFIC_BAR
    [self.trafficBar updateNavRoute:manager.curNavRoute];
#endif
}

- (void)driveNavManager:(TNKDriveNavManager *)manager updateNavigationData:(TNKDriveNavigationData *)navigationData {
    
#if CUSTOM_TRAFFIC_BAR
    // 更新剩余距离
    [self.trafficBar updateRemainingDistance:navigationData.totalDistanceLeft];
#endif

}

// 路况更新
- (void)driveNavManagerDidUpdateRotueTraffic:(TNKDriveNavManager *)manager {
#if CUSTOM_TRAFFIC_BAR
    [self.trafficBar updateNavRoute:manager.curNavRoute];
#endif

}

- (void)driveNavManager:(TNKDriveNavManager *)manager
         routeDidChange:(TNKDriveNavRoute *)navRoute
         changeRouteMsg:(TNKChangeRouteMsg *)msg {
#if CUSTOM_TRAFFIC_BAR
    // 路线发生切换
    [self.trafficBar updateNavRoute:manager.curNavRoute];
#endif
}


#pragma mark - TNKDriveNavViewDelegate
- (void)navViewDidShowEnlargedMap:(TNKDriveNavView *)navView {
    // 展示了放大图，先让面板透明
   
    self.trafficBar.hidden = YES;
}

- (void)navViewDidHideEnlargedMap:(TNKDriveNavView *)navView {
    // 隐藏了放大图，恢复面板
    self.trafficBar.hidden = NO;
}

@end
