//
//  DriveCustomTrafficBar.m
//  NavKitDemo
//
//  Created by mol on 2023/2/9.
//

#import "DriveCustomTrafficBar.h"

@interface DriveCustomTrafficBar ()

// 路况颜色数组
@property (nonatomic, copy) NSArray<UIView *> *trafficColorViews;

// 走过时覆盖为灰色
@property (nonatomic, strong) UIView *colorBgView;

// 走过时覆盖为灰色
@property (nonatomic, strong) UIView *passedColorView;

// 小车的车标
@property (nonatomic, strong) UIImageView *carIconView;

// 记录总长度，单位米
@property (nonatomic, assign) int totalDistance;

@end

@implementation DriveCustomTrafficBar

#pragma mark - lifecycle

- (instancetype)initWithFrame:(CGRect)frame {
    self = [super initWithFrame:frame];
    if (self) {
        
        [self setup];
    }
    
    return self;
}

- (void)setup {
    
    //
    self.layer.cornerRadius = self.bounds.size.width / 2;
    self.layer.borderColor = [UIColor whiteColor].CGColor;
    self.layer.borderWidth = 3;
    
    // 取导航默认的颜色，也可以自定义颜色
    self.backgroundColor = TNKRouteTrafficStatusColor(TNKRouteTrafficDataStatusSmooth);
    
    self.colorBgView = [[UIView alloc] initWithFrame:self.bounds];
    self.colorBgView.layer.cornerRadius = self.colorBgView.bounds.size.width / 2;
    self.colorBgView.clipsToBounds = YES;
    [self addSubview:self.colorBgView];

    self.passedColorView = [[UIView alloc] initWithFrame:CGRectMake(0, 0, self.bounds.size.width, 0)];
    self.passedColorView.autoresizingMask = UIViewAutoresizingFlexibleWidth;
    self.passedColorView.backgroundColor = TNKRouteTrafficStatusColor(TNKRouteTrafficDataStatusPassed);
    
    [self.colorBgView addSubview:self.passedColorView];
    
    
    self.carIconView = [[UIImageView alloc] initWithFrame:CGRectMake(0, 0, 42, 42)];
    self.carIconView.contentMode = UIViewContentModeScaleAspectFill;
    self.carIconView.image = [UIImage imageNamed:@"trafficbar_ic_car"];
    self.carIconView.hidden = YES;
    [self addSubview:self.carIconView];
}


#pragma mark - public

// 更新路线以及路况
- (void)updateNavRoute:(TNKDriveNavRoute *)navRoute {
    // 如果考虑偏航前走过的距离，就加上distBetweenOrginAndRouteFistPoint；否则, 不用加。
    self.totalDistance = navRoute.searchRoute.totalDistance + navRoute.distBetweenOrginAndRouteFistPoint;
    self.carIconView.hidden = NO;
    
    // 移除之前的颜色
    for (UIView *colorView in self.trafficColorViews) {
        [colorView removeFromSuperview];
    }
    
    if (navRoute.searchRoute.totalDistance > 0) {
        // 更新光柱图颜色
        NSMutableArray<UIView *> *tmpViews = [NSMutableArray arrayWithCapacity:navRoute.curTrafficItems.count];
        CGFloat top = 0;
        for (int i = 0; i < navRoute.curTrafficItems.count; i++) {
            TNKRouteTrafficItem *trafficItem = navRoute.curTrafficItems[navRoute.curTrafficItems.count - i - 1];
            CGFloat height = trafficItem.distance * 1.0 / navRoute.searchRoute.totalDistance * self.bounds.size.height;
            height = MIN(self.bounds.size.height - top, height);
            UIView *colorView = [[UIView alloc] initWithFrame:CGRectMake(0, top, self.bounds.size.width, height)];
            colorView.backgroundColor = TNKRouteTrafficStatusColor(trafficItem.trafficStatus);
            [self.colorBgView insertSubview:colorView belowSubview:self.passedColorView];
            [tmpViews addObject:colorView];
            top += height;

        }
        self.trafficColorViews = tmpViews;
    }
    
    // 更新走过区域以及小车位置
    [self updateRemainingDistance:navRoute.remainingDistance];
}


// 更新剩余距离
- (void)updateRemainingDistance:(int)remainingDistance {
    
    // 设置已走过区域
    CGFloat height = 0;
    if (self.totalDistance > 0) {
        remainingDistance = MAX(0, MIN(self.totalDistance, remainingDistance));
        height = self.bounds.size.height * (self.totalDistance - remainingDistance) / self.totalDistance;
    }
    self.passedColorView.frame = CGRectMake(0, self.bounds.size.height - height, self.bounds.size.width, height);
    
    // 更新小车位置
    self.carIconView.frame = CGRectMake((self.bounds.size.width - self.carIconView.bounds.size.width) / 2,
                                        self.bounds.size.height - height - self.carIconView.bounds.size.height / 2,
                                        self.carIconView.bounds.size.width,
                                        self.carIconView.bounds.size.height);
}

@end
