//
//  DriveCustomPanelController.m
//  NavKitDemo
//
//  Created by mol on 2023/2/8.
//

#import "DriveCustomPanelController.h"
#import "DriveCustomPanel.h"
#import "DeviceUtil.h"

@interface DriveCustomPanelController ()

// 自定义导航面板
@property (nonatomic, strong) DriveCustomPanel *customPanel;
// 自定义车道线
@property (nonatomic, strong) UIImageView *customLaneView;

@end

@implementation DriveCustomPanelController

- (void)viewDidLoad {
    [super viewDidLoad];
    
    // 隐藏默认的导航面板以及车道线,改为自定义
    self.navView.navigationPanelEnabled = NO;
    self.navView.guidedLaneEnabled = NO;
    
    // 发起算路
    [self planRoutes];
    
    [self setupCustomPanel];
}

#pragma mark - private

/// 初始化自定义面板
- (void)setupCustomPanel {
    CGFloat leftMargin = 10;
    self.customPanel = [DriveCustomPanel CustomPanel];
    self.customPanel.frame = CGRectMake(leftMargin,
                                        [DeviceUtil topPadding] + 35,
                                        self.view.bounds.size.width - 2 * leftMargin,
                                        140);
    self.customPanel.autoresizingMask = UIViewAutoresizingFlexibleWidth;
    // 先隐藏，导航开启再显示
    self.customPanel.hidden = YES;
    [self.view addSubview:self.customPanel];
    
    self.customLaneView = [[UIImageView alloc] initWithFrame:CGRectMake(self.customPanel.frame.origin.x,
                                                                        CGRectGetMaxY(self.customPanel.frame) + 8,
                                                                        self.customPanel.frame.size.width,
                                                                        50)];
    self.customLaneView.autoresizingMask = UIViewAutoresizingFlexibleWidth;
    self.customLaneView.contentMode = UIViewContentModeScaleAspectFit;
    // 先隐藏，有车道线的时候再展示
    self.customLaneView.hidden = YES;
    [self.view addSubview:self.customLaneView];

    // 修改元素顶部间距
    self.navView.panelTopGap = self.customPanel.frame.origin.y;
}

/// 初始路径规划
- (void)planRoutes {
    
    __weak typeof(self) weakself = self;
    
    TNKDriveRouteRequest *request = [[TNKDriveRouteRequest alloc] init];
    
    // 起点信息
    TNKSearchNavPoint *origin = [[TNKSearchNavPoint alloc] init];
    origin.coordinate = CLLocationCoordinate2DMake(40.038591, 116.273023);
    
    // 终点信息. 可以使用关键词检索获得坐标、poi id、名称等信息。
    // 详情见: https://lbs.qq.com/service/webService/webServiceGuide/webServiceSuggestion
    TNKSearchNavPoint *destination = [[TNKSearchNavPoint alloc] init];
    destination.coordinate = CLLocationCoordinate2DMake(39.907773, 116.401273);
    destination.poiID = @"2199027907804";
    destination.title = @"天安门东[地铁站]";
    
    // 赋值
    request.origin = origin;
    request.destination = destination;
    
    // 算路策略
    TNKDriveRouteSearchOption *searchOption = [[TNKDriveRouteSearchOption alloc] init];
    // 导航场景. 1表示去接乘客, 2表示去送乘客. 默认值为0.
    searchOption.navScene = 1;
    request.searchOption = searchOption;

    // 初始路径规划
    [self.navManager searchRoutesWithRequest:request
                                  completion:^(TNKDriveRouteSearchResult * _Nullable searchResult,
                                               NSError * _Nullable error) {
       
        if (error != nil) {
            // 算路失败
            [weakself showAlertWithError:error];
            return;
        }
        
        // 算路成功，开始模拟导航
        [weakself startNav];
    }];
}

- (void)startNav {
    // 使用第一条路线进行模拟导航
    self.navManager.locationSimulator.enabled = YES;
    [self.navManager startNavWithIndex:0];
}

#pragma mark - TNKDriveNavDelegate
- (void)driveNavManager:(TNKDriveNavManager *)manager updateNavigationData:(TNKDriveNavigationData *)navigationData {
    // 更新导航数据，显示自定义面板
    self.customPanel.hidden = NO;
    
    // 更新数据
    [self.customPanel updateNavigationData:navigationData];
}

- (void)driveNavManager:(TNKDriveNavManager *)manager showGuidedLaneInfo:(TNKLaneInfo *)laneInfo {
    // 显示车道线
    self.customLaneView.hidden = NO;
    self.customLaneView.image = laneInfo.image;
}

- (void)driveNavManagerHideGuidedLaneInfo:(TNKDriveNavManager *)manager {
    // 隐藏车道线
    self.customLaneView.hidden = YES;
    self.customLaneView.image = nil;
}


#pragma mark - TNKDriveNavViewDelegate
- (void)navViewDidShowEnlargedMap:(TNKDriveNavView *)navView {
    // 展示了放大图，先让面板透明
    self.customPanel.alpha = 0;
    
    // 车道线放在放在放大图底部之上
    self.customLaneView.frame = CGRectMake(self.customPanel.frame.origin.x,
                                           CGRectGetMaxY(self.navView.enlargedMapConfig.enlargedMapRect) - self.customLaneView.frame.size.height,
                                           self.customLaneView.frame.size.width,
                                           self.customLaneView.frame.size.height);
    
}

- (void)navViewDidHideEnlargedMap:(TNKDriveNavView *)navView {
    // 隐藏了放大图，恢复面板
    self.customPanel.alpha = 1;
    
    // 恢复车道线位置，放在面板下边
    self.customLaneView.frame = CGRectMake(self.customPanel.frame.origin.x,
                                           CGRectGetMaxY(self.customPanel.frame) + 8,
                                           self.customLaneView.frame.size.width,
                                           self.customLaneView.frame.size.height);
    
}

@end
