//
//  BaseCaseController.m
//  NaviKitDemo
//
//  Created by mol on 2022/12/16.
//

#import "DriveBaseCaseController.h"
#import <TencentNavKit/TencentNavKit.h>
#import <TNKAudioPlayer/TNKAudioPlayer.h>
#import "RoutePolyline.h"
#import "DeviceUtil.h"
#import "WebRoutePolyline.h"

@interface DriveBaseCaseController ()

/// 导航管理类
@property (nonatomic, strong) TNKDriveNavManager *navManager;
/// 导航地图
@property (nonatomic, strong) TNKDriveNavView *navView;

// 开启导航前所画的路线。 导航过程中无需开发者画路线
@property (nonatomic, strong) NSMutableArray<RoutePolyline *> *routePolylines;
// 开启导航前所画的起终点marker。 导航过程中无需开发者画
@property (nonatomic, strong) id<QAnnotation> destCircleAnnotation;
@property (nonatomic, strong) id<QAnnotation> originCircleAnnotation;

// 初始路径规划的结果
@property (nonatomic, strong, nullable) TNKDriveRouteSearchResult *routesRsp;
// 当前算中的路线
@property (nonatomic, assign) NSInteger selectedIdx;

@end

@implementation DriveBaseCaseController

#pragma mark - liefcycle

- (void)dealloc {
    NSLog(@"%@_dealloc", NSStringFromClass(self.class));
    [self.navManager unregisterNavDelegate:self.navView];
    [self.navManager unregisterNavDelegate:self];
    [[TNKDriveNavManager sharedInstance] stopNav];
}

- (void)viewDidLoad {
    [super viewDidLoad];
    // Do any additional setup after loading the view.
    self.routePolylines = [NSMutableArray array];
    
    // 创建驾车导航地图
    self.navView = [[TNKDriveNavView alloc] initWithFrame:self.view.bounds];
    self.navView.autoresizingMask = UIViewAutoresizingFlexibleWidth | UIViewAutoresizingFlexibleHeight;
    [self.view addSubview:self.navView];
    self.navView.delegate = self;
    self.navView.showUIElements = NO;
    self.mapView.delegate = self;
    [self.mapView setCenterOffsetV2:CGPointMake(0.5, 0.5)];

    // 创建驾车管理对象
    self.navManager = [TNKDriveNavManager sharedInstance];
    self.navManager.locationSimulator.simulationSpeed = 60;
    // 使用tts模块
    self.navManager.audioPlayer = [TNKAudioPlayer sharedAudioPlayer];
    // 注册回调，保证导航地图接收导航信息
    [self.navManager registerNavDelegate:self.navView];
    // 开发者监听导航回调
    [self.navManager registerNavDelegate:self];
    // 导航中是否展示多路线
    self.navManager.multiRoutesEnabled = YES;
    // 开启导航时，是否将初始路径规划的其他路线展示出来
    self.navManager.showMultiRoutesWhenStartNav = YES;
}

#pragma mark - public

- (QMapView *)mapView {
    return self.navView.mapView;
}

- (void)clearRoutePlan {
    
    [self.mapView removeOverlays:self.routePolylines];
    [self.routePolylines removeAllObjects];
    self.routesRsp = nil;
    [self.mapView removeAnnotation:self.destCircleAnnotation];
    [self.mapView removeAnnotation:self.originCircleAnnotation];
}

- (void)drawRoutePlan:(TNKDriveRouteSearchResult *)routesRsp
        selectedIndex:(NSInteger)selectedIndex {
    
    self.mapView.showsUserLocation = NO;
    self.routesRsp = routesRsp;
        
    for (int i = 0; i < routesRsp.routes.count; i++) {
        TNKDriveRoute *route = routesRsp.routes[i];
        
        // 创建路线,一条路线由一个点数组和线段数组组成
        RoutePolyline *polyline = [[RoutePolyline alloc] initWithDriveRoute:route];
        
        if (selectedIndex == i) {
            polyline.selectedLine = YES;
        }

        [self.mapView addOverlay:polyline];
        [self.routePolylines addObject:polyline];
    }
    
    self.selectedIdx = selectedIndex;
  
    // 调整视野
    [self adjustRoutesFitMap];
}

- (void)adjustRoutesFitMap {
    
    NSMutableArray<TNKCoordinatePoint *> *points = [NSMutableArray array];
    
    for (int i = 0; i < self.routesRsp.routes.count; i++) {
        TNKDriveRoute *route = self.routesRsp.routes[i];
        [points addObjectsFromArray:route.routePoints];
    }
    
    CGFloat topPadding = [DeviceUtil topPadding] + 30;
    CGFloat bottomPadding = [DeviceUtil bottomPadding] + 30;

    QMapPoint firstMapPoint = QMapPointForCoordinate(points[0].coordinate);
    
    CGFloat minX = firstMapPoint.x;
    CGFloat minY = firstMapPoint.y;
    CGFloat maxX = minX;
    CGFloat maxY = minY;
    
    for (int i = 1; i < points.count; i++) {
        
        QMapPoint point = QMapPointForCoordinate(points[i].coordinate);
        
        if (point.x < minX) {
            minX = point.x;
        }
        if (point.y < minY) {
            minY = point.y;
        }
        if (point.x > maxX) {
            maxX = point.x;
        }
        if (point.y > maxY) {
            maxY = point.y;
        }
    }
    
    CGFloat width  = fabs(maxX - minX);
    CGFloat height = fabs(maxY - minY);
    
    QMapRect visibleRect = QMapRectMake(minX, minY, width, height);
    [self.mapView setVisibleMapRect:visibleRect
                        edgePadding:UIEdgeInsetsMake(topPadding, 30, bottomPadding, 30)
                           animated:YES];
}

- (void)showAlertWithError:(NSError *)error {
    
    UIAlertAction *okAction = [UIAlertAction actionWithTitle:@"OK"
                                                       style:UIAlertActionStyleDefault
                                                     handler:^(UIAlertAction * _Nonnull action) {
    }];
    
    UIAlertController *alertController = [UIAlertController alertControllerWithTitle:@"失败"
                                                                             message:[NSString stringWithFormat:@"%d:%@", (int)error.code, error.userInfo[NSLocalizedDescriptionKey]]
                                                                      preferredStyle:UIAlertControllerStyleAlert];
    [alertController addAction:okAction];
    [self presentViewController:alertController animated:YES completion:nil];
}

#pragma mark - QMapViewDelegate

- (QOverlayView *)mapView:(QMapView *)mapView viewForOverlay:(id<QOverlay>)overlay {
    if ([overlay isKindOfClass:[RoutePolyline class]]) {
        RoutePolyline *tl = (RoutePolyline*)overlay;
        
        QTexturePolylineView *polylineRender = [[QTexturePolylineView alloc] initWithPolyline:overlay];
        polylineRender.drawType = QTextureLineDrawType_ColorLine;
        polylineRender.drawSymbol = YES;
        polylineRender.segmentColor = tl.arrLine;
        polylineRender.lineWidth   = 10;
        polylineRender.borderWidth = 1.5;
        polylineRender.text = tl.texts;
        polylineRender.zIndex = tl.selectedLine? 1 : 0;
        return polylineRender;
    } else if ([overlay isKindOfClass:[QPolygon class]]) {
        QPolygonView *polygonRender = [[QPolygonView alloc] initWithPolygon:overlay];
        // 设置线宽
        polygonRender.lineWidth   = 3;
        // 设置描边颜色
        polygonRender.strokeColor = [UIColor colorWithRed:.2 green:.1 blue:.1 alpha:.8];
        // 设置填充颜色
        polygonRender.fillColor   = [[UIColor blueColor] colorWithAlphaComponent:0.2];
        
        return polygonRender;
    } else if ([overlay isKindOfClass:[WebRoutePolyline class]]) {
        
        WebRoutePolyline *poyline = (WebRoutePolyline*)overlay;
        
        QTexturePolylineView *polylineRender = [[QTexturePolylineView alloc] initWithPolyline:overlay];
        polylineRender.drawType = QTextureLineDrawType_ColorLine;
        polylineRender.drawSymbol = YES;
        polylineRender.segmentColor = poyline.arrLine;
        polylineRender.lineWidth   = 10;
        polylineRender.borderWidth = 1.5;
        polylineRender.zIndex = poyline.selectedLine? 1 : 0;
        return polylineRender;
    }
    
    return nil;
}

- (void)mapView:(QMapView *)mapView didTapOverlay:(id<QOverlay>)overlay {
    
}

#pragma mark - TNKDriveNavDelegate
- (void)driveNavManager:(TNKDriveNavManager *)manager didStartRerouting:(TNKRerouteReason)rerouteReason {
    NSLog(@"didStartRerouting?isRunning=%d", manager.isRunning);
}


- (void)driveNavManagerDidStart:(TNKDriveNavManager *)manager {
    NSLog(@"driveNavManagerDidStart?isRunning=%d", manager.isRunning);

}

- (void)driveNavManagerDidStop:(TNKDriveNavManager *)manager {
    NSLog(@"driveNavManagerDidStop?isRunning=%d", manager.isRunning);

}

- (void)driveNavManagerWillArriveDestination:(TNKDriveNavManager *)manager {
    NSLog(@"driveNavManagerWillArriveDestination?isRunning=%d", manager.isRunning);

}

#pragma mark - TNKDriveNavViewDelegate
- (void)navViewCloseButtonClicked:(TNKBaseNavView *)navView {
    [self.navManager stopNav];
    [self.navView clearAllRouteUI];
    self.navView.showUIElements = NO;
    self.navView.navigationPanelEnabled = NO;
    self.navView.guidedLaneEnabled = NO;
}

@end
