//
//  DistrictChildrenSearchViewController.m
//  QMapKitDemo
//
//  Created by Keith Cao on 2023/10/20.
//  Copyright © 2023 tencent. All rights reserved.
//

#import "DistrictChildrenSearchViewController.h"
#import <QMapKit/QMSSearchKit.h>
#import "QSearchTools.h"

@interface DistrictChildrenSearchViewController ()<QMSSearchDelegate>

@property (nonatomic, strong) QMSSearcher *searcher;

@end

@implementation DistrictChildrenSearchViewController

- (void)viewDidLoad {
    [super viewDidLoad];
    
	[self executeDistrictChildSearch];
}

#pragma mark -- Search Action

- (void)executeDistrictChildSearch
{
	self.searcher = [[QMSSearcher alloc] initWithDelegate:self];
	
	QMSDistrictChildrenSearchOption *districtSearchOption = [[QMSDistrictChildrenSearchOption alloc] init];
	districtSearchOption.ID = @"110000";	/// 行政区划代码（支持多个代码）
	districtSearchOption.get_polygon = QMSDistrictPolygonWithLandOnly;
	districtSearchOption.max_offset = 30000;
	
	[self.searcher searchWithDistrictChildrenSearchOption:districtSearchOption];
}

- (void)dealWithDistrictSearchResult:(QMSDistrictSearchResult *)result
{
	NSArray <QMSDistrictData *> *districtArray = result.result.firstObject;
	
	if (districtArray.count > 0)
	{
		QMapRect visibleRect = QMapRectNull;
		// 遍历每个行政区划数据
		for (QMSDistrictData *data in districtArray)
		{
			// 遍历该行政区划下的轮廓数据
			for (NSArray <NSValue *> *polyonPoints in data.polygons)
			{
				QMapRect rect = [self createPolygonOverlayByDistrictData:polyonPoints];
				
				// 区域外接矩形集合
				visibleRect = QMapRectUnion(visibleRect, rect);
			}
		}
		
		[self.mapView setVisibleMapRect:visibleRect animated:YES];
	}
	
}

#pragma mark -- Search Delegate

- (void)searchWithSearchOption:(QMSSearchOption *)searchOption didFailWithError:(NSError *)error
{
	NSLog(@"search error :%@", error);
}

- (void)searchWithDistrictSearchOption:(QMSDistrictBaseSearchOption *)districtSearchOption didRecevieResult:(QMSDistrictSearchResult *)districtSearchResult
{
	[self dealWithDistrictSearchResult:districtSearchResult];
}

#pragma mark -- MapDelegate

- (QOverlayView *)mapView:(QMapView *)mapView viewForOverlay:(id<QOverlay>)overlay
{
	if ([overlay isKindOfClass:[QPolygon class]])
	{
		QPolygonView *render = [[QPolygonView alloc] initWithPolygon:overlay];
		render.lineWidth   = 3;
		render.strokeColor = [UIColor colorWithRed:70/255.0 green:119/255.0 blue:223/255.0 alpha:.8];
		render.fillColor   = [[UIColor grayColor] colorWithAlphaComponent:0.2];
		
		return render;
	}
	return nil;
}

#pragma mark -- Tools

- (QMapRect)createPolygonOverlayByDistrictData:(NSArray <NSValue *> *)points
{
	// 行政区划轮廓的点数组元素为型为encode的CLLocationCoordinate2D
	int count = (int)points.count;
	
	if (count < 3)
	{
		return QMapRectNull;
	}
	
	CLLocationCoordinate2D coords[count];
	
	for(int i = 0; i < count; i++)
	{
		coords[i] = [QSearchTools decodeCoordinateValue:points[i]];
	}
	
	QPolygon *polygon = [[QPolygon alloc] initWithWithCoordinates:coords count:count];
	
	[self.mapView addOverlay:polygon];
	
	return polygon.boundingMapRect;
}

@end
