//
//  CircleOverlayViewController.m
//  QMapKitDemo
//
//  Created by Keith Cao on 2023/10/17.
//  Copyright © 2023 tencent. All rights reserved.
//

#import "CircleOverlayViewController.h"

typedef NS_ENUM(NSUInteger, TestCircleType)
{
	TypeDefaultCircle   = 0,
	TypeDashCircle,
	TypeDotCircle,
	TypeCircleMAX
};

@interface CircleOverlayViewController ()

@property (nonatomic, strong) NSMutableArray *overlays;

@end

@implementation CircleOverlayViewController

- (void)handleTestAction
{
	static int cn = 0;
	
	[self.overlays enumerateObjectsUsingBlock:^(id<QOverlay>  _Nonnull obj, NSUInteger idx, BOOL * _Nonnull stop) {
		
		QOverlayView *view = [self.mapView viewForOverlay:obj];
		NSAssert(view, @"view is not null");
		
		if ([view isKindOfClass:[QCircleView class]])
		{
			QCircleView *circleView = (QCircleView *)view;
			
			circleView.lineType = cn % TypeCircleMAX;
			circleView.lineDashPattern = @[@(10), @(10)];
		}
	}];
	
	cn++;
}

#pragma mark - CircleView

- (QCircleView *)createCircleView:(QCircle *)circle
{
	QCircleView *view = [[QCircleView alloc] initWithCircle:circle];
	view.fillColor = [[UIColor blueColor] colorWithAlphaComponent:0.3];
	view.strokeColor = [[UIColor greenColor] colorWithAlphaComponent:1];
	view.lineWidth = 12;
	
	return view;
}

- (QCircleView *)createDashCircleView:(QCircle *)circle
{
	QCircleView *view = [[QCircleView alloc] initWithCircle:circle];
	view.fillColor = [[UIColor blueColor] colorWithAlphaComponent:0.3];
	view.strokeColor = [[UIColor greenColor] colorWithAlphaComponent:1];
	view.lineWidth = 12;
	view.lineDashPattern = @[@(12), @(2)];
	
	view.lineType = QOverlayStrokeType_Dash;
	
	return view;
}

- (QCircleView *)createDotCircleView:(QCircle *)circle
{
	QCircleView *view = [[QCircleView alloc] initWithCircle:circle];
	view.fillColor = [[UIColor blueColor] colorWithAlphaComponent:0.3];
	view.strokeColor = [[UIColor greenColor] colorWithAlphaComponent:1];
	view.lineWidth = 12;
	
	view.lineType = QOverlayStrokeType_Dot;
	view.lineDashPattern = @[@(12)];
	
	return view;
}

#pragma mark - QMapViewDelegate

- (QOverlayView *)mapView:(QMapView *)mapView viewForOverlay:(id<QOverlay>)overlay
{
	if ([overlay isKindOfClass:[QCircle class]])
	{
		QCircle *circle = (QCircle *)overlay;
		TestCircleType type = [circle.userData intValue];
		QOverlayView *view = nil;
		
		if (type == TypeDashCircle)
		{
			view = [self createDashCircleView:circle];
		}
		else if (type == TypeDotCircle)
		{
			view = [self createDotCircleView:circle];
		}
		else
		{
			view = [self createCircleView:circle];
		}
		
		return view;
	}
	
	return nil;
}


#pragma mark - Setup

- (void)setupOverlays
{
	self.overlays = [NSMutableArray array];
	
	CLLocationCoordinate2D coordinates = CLLocationCoordinate2DMake(39.9442, 116.514);
	
	for (int i = 0; i < TypeCircleMAX; i++)
	{
		coordinates.longitude += 0.4;
		QCircle *circle = [QCircle circleWithCenterCoordinate:coordinates radius:10000];
		circle.userData = @(i);
		[self.overlays addObject:circle];
	}
}

- (void)viewDidLoad
{
	[super viewDidLoad];
	
	[self setupOverlays];
	
	[self.mapView addOverlays:self.overlays];
	
	QMapRect mapRect = QMapRectNull;
	for (int i = 0; i < self.overlays.count; i++)
	{
		if (QMapRectIsEmpty(mapRect))
		{
			mapRect = [self.overlays[i] boundingMapRect];
		}
		else
		{
			mapRect = QMapRectUnion(mapRect, [self.overlays[i] boundingMapRect]);
		}
	}
	
	[self.mapView setVisibleMapRect:mapRect edgePadding:UIEdgeInsetsMake(20, 40, 20, 40) animated:YES];
}
@end
