//
//  QModelOverlay.h
//  QMapKit
//
//  Created by fan on 2020/7/11.
//  Copyright © 2020 tencent. All rights reserved.
//

#import <QMapKit/QGLOverlay.h>
#import <QMapKit/QGeometry.h>

NS_ASSUME_NONNULL_BEGIN

@class QGLOverlayRenderMode;
@class QGLOverlayCustomMaskMode;
/**
* @brief        模型类型.
*/
typedef NS_ENUM(NSInteger, QModelRenderType) {
    QGeographicUnit,     ///< 地理单位
    QScreenUnit          ///< 屏幕单位
};

/**
* @brief        模型动画类型.
*/
typedef NS_ENUM(NSInteger, QModelAnimationType) {
    QModelAnimation_None,        ///< 无动画
    QModelAnimation_Grow         ///< 生长/拍平动画
};


/**
 * @brief QModelOverlayDelegate: QGLOverlay的Delegate
 **/
@protocol QModelOverlayDelegate <QGLOverlayDelegate>

@optional

/**
 *  @brief  glOverlay平移动画完成后会调用此接口
 *
 *  @param glOverlay  GLOverlay
 *  @param finished  动画是否完成
*/
- (void)glOverlay:(QGLOverlay *)glOverlay translateAnimationDidComplete:(BOOL)finished;

@end

/**
* @brief        模型覆盖物
*/
@interface QModelOverlay : QGLOverlay

/**
 * @brief 设置可视化覆盖物的渲染模式，默认为：QGLOverlayGeoRenderMode
 */
@property (nonatomic, strong) QGLOverlayRenderMode * renderMode;

/**
* @brief    模型文件的全路径(请将文件放到同一目录下)
*/
@property (nonatomic, copy ) NSString   *path;

/**
* @brief    模型原点对应的经纬坐标
*/
@property (nonatomic, assign) CLLocationCoordinate2D   coordinate;
/**
* @brief    模型原点对应的高度。（单位：米）
*/
@property (nonatomic, assign) double   altitude;

/**
* @brief    x轴的旋转角度
*/
@property (nonatomic, assign) CGFloat   rotationX;
/**
* @brief    y轴的旋转角度
*/
@property (nonatomic, assign) CGFloat   rotationY;
/**
* @brief    z轴的旋转角度
*/
@property (nonatomic, assign) CGFloat   rotationZ;

/**
* @brief    动画类型.
* 0: 无动画。1：拔起。拍平
*/
@property (nonatomic, assign) QModelAnimationType animateType;

/**
* @brief    曝光度。默认1.0
*/
@property (nonatomic, assign) CGFloat   exposure;

/**
* @brief    设置是否为无光照模式。默认为NO
*/
@property (nonatomic, assign, getter=isUnlit) BOOL unlit;

@end


#pragma mark Model Animation

/// ModelOverlay内置动画的信息
@interface QModelAnimationInfo : NSObject

/**
* @brief    动画标识
*/
@property (nonatomic, assign) NSUInteger    index;
/**
* @brief    动画名字.可能为空
*/
@property (nonatomic, copy) NSString      *name;
/**
* @brief    动画时长. 秒
*/
@property (nonatomic, assign) double        duration;

@end

@interface QModelOverlay (Animation)
/**
 *  @brief 图层平移动画接口, 仅支持模型可视化数据图层
 *  @param positions    图层轨迹点 [lat, lng, lat, lng...]
 *  @param duration      动画持续时间, 单位: 秒
 *  @param rotateOffset     初始旋转角度，单位角度
 *  @param enableRotate   是否需要自动旋转
 */
- (void)startTranslateAnimation:(NSArray<NSNumber *> *)positions duration:(CGFloat)duration initRotation:(CGFloat)rotateOffset autoRotate:(BOOL)enableRotate;

/**
* @brief    获取内置动画信息
*/
- (NSArray<QModelAnimationInfo*>*)getAnimationInformation;
/**
* @brief    播放内置动画
*/
- (void)playAnimationWithIndex:(int)index speed:(float)speed autoRepeat:(BOOL)autoRepeat;
/**
* @brief    停止当前内置动画
*/
- (void)stopAnimation;

@end

/**
 * @brief gltf模型KHRMaterialVariants材质信息
 */
@interface QModelVariantInfo : NSObject

/**
 * @brief 材质名字
 */
@property (nonatomic, copy) NSString *name;

@end

/**
 * @brief gltf模型自定义颜色和KHRMaterialVariants材质
 */
@interface QModelOverlay (CustomMask)

/**
 * @brief gltf模型自定义KHRMaterialVariants材质(需在模型加载完成调用)
 * @param index 自定义材质的下标
 */
- (void)setMaterialVariant:(NSUInteger)index;

/**
 * @brief 获取gltf模型当前的KHRMaterialVariants材质下标 (需在模型加载完成调用)
 */
- (int)getCurrentMaterialVariant;

/**
 * @brief 获取gltf模型支持的KHRMaterialVariants材质信息 (需在模型加载完成调用)
 */
- (NSArray <QModelVariantInfo *> *)getMaterialVariantsInfo;

/**
 * @brief 模型自定义颜色(需在模型加载完成调用)
 * @param color 自定义颜色
 */
- (void)setMonoColor:(UIColor *)color;

/**
 * @brief 重置模型自定义颜色(需在模型加载完成调用)
 */
- (void)resetMonoColor;

@end

/**
 * @brief 可视化覆盖物的渲染模式基类
 */
@interface QGLOverlayRenderMode : NSObject

@end

/**
 * @brief 可视化覆盖物的像素单位渲染模式
 */
@interface QGLOverlayPixelRenderMode : QGLOverlayRenderMode

/**
 * @brief 可视化覆盖物的像素渲染模式范围区域，单位：point，默认 32 x 32 
 */
@property (nonatomic, assign) CGSize size;

@end

/**
 * @brief 可视化覆盖物的地理单位渲染模式
 */
@interface QGLOverlayGeoRenderMode : QGLOverlayRenderMode

/**
 * @brief 缩放比例，默认值: 1
 */
@property(nonatomic, assign) CGFloat scale;

@end

NS_ASSUME_NONNULL_END
