//
//  QMarkerView.h
//  QMapKit
//
//  Created by Keith Cao on 2024/1/29.
//  Copyright © 2024 tencent. All rights reserved.
//

#import <UIKit/UIKit.h>
#import "QMarker.h"
#import "QOverlayView.h"
#import "QMarkerViewDefines.h"
#import "QMarkerViewAnimationLayer.h"

NS_ASSUME_NONNULL_BEGIN


/// 此类是QMarker用于显示标记覆盖物的view
@interface QMarkerView : QOverlayView

/**
 *	@brief markerView初始化接口
 *	@param	marker QMarker类对象
 */
- (instancetype)initWithMarker:(QMarker *)marker;

/**
 *	@brief marker 是否可见, 默认: NO
 */
@property (nonatomic, assign) BOOL isHidden;

/**
 * @brief marker 的锚点, 根据marker图片的宽高进行等比例平移，默认值[0.5, 0.5]
 *  x 左右平移比例, 取值范围 [0, 1]
 *  y 上下平移比例, 取值范围 [0, 1]
 */
@property (nonatomic, assign) CGPoint centerOffset;

/**
 * @brief marker 的透明度, 取值范围[0,1] 默认：1
 */
@property (nonatomic, assign) CGFloat alpha;

/**
 * @brief marker 的缩放比例值,  默认：[1,1]
 *   x 表示 宽度比例, 有效值>=0
 *   y 表示 高度比例, 有效值>=0
 */
@property (nonatomic, assign) CGPoint scale;

/**
 * @brief marker 的旋转角度值,  默认：0
 */
@property (nonatomic, assign) CGFloat rotation;

/**
 * @brief marker 的标题
 */
@property (nonatomic, nullable, copy) NSString *title;

/**
 * @brief marker 的副标题
 */
@property (nonatomic, nullable, copy) NSString *subtitle;

/**
 *  @brief 覆盖物交互开关, 默认: YES
 */
@property (nonatomic, assign) BOOL userInteractionEnabled;

/**
 *  @brief 覆盖物拖拽开关, 默认: NO
 */
@property (nonatomic, assign) BOOL draggable;

/**
 *	@brief marker 的图标
 *  图标最大尺寸限制为 1024 * 1024像素，超出时按等比例缩小至有效尺寸内
 */
@property (nonatomic, nullable, strong) UIImage *image;

/**
 * @brief QMarkerView的碰撞类型
 */
@property (nonatomic, assign) QMarkerViewCollisionType collisionType;

/**
 * @brief QMarkerView的拖动状态
 */
@property (nonatomic, assign) QMarkerViewDragState dragState;

/**
 * @brief 是否允许展示callout, 默认值 NO
 */
@property (nonatomic, assign) BOOL canshowCallout;

/**
 *  @brief  自定义UIView类型的callout, 赋值后，优先展示
 */
@property (nonatomic, strong, nullable) UIView *customCalloutView;

/// 更新MarkerView的中心点
/// @param location 中心点坐标
- (void)updateCenterCoordinate:(CLLocationCoordinate2D)location;

/// 是否展示callout
/// @param toShow 是否展示
- (void)showCalloutView:(BOOL)toShow;


/// 刷新callout
- (void)refreshCalloutView;

/**
 * @brief QMarkerView的动画layer
 */
@property (nonatomic, strong, readonly) QMarkerViewAnimationLayer *animationLayer;

@end

NS_ASSUME_NONNULL_END
