//
//  OfflineDownloadCell.m
//  QMapKitDemo
//
//  Created by v_hefang on 2020/5/29.
//  Copyright © 2020 tencent. All rights reserved.
//

#import "OfflineDownloadCell.h"

@interface OfflineDownloadCell ()
@property (nonatomic, strong) UILabel *titleLabel;
@property (nonatomic, strong) UILabel *sizeLabel;
@property (nonatomic, strong) UIButton *downloadButton;
@property (nonatomic, strong) UIButton *cancelButton;
@property (nonatomic, strong) UIButton *deleteButton;
@end

@implementation OfflineDownloadCell

- (UIButton *)downloadButton {
    if (_downloadButton == nil) {
        _downloadButton = [UIButton buttonWithType:UIButtonTypeSystem];
        [_downloadButton setTitle:@"下载" forState:UIControlStateNormal];
        [_downloadButton setTitleColor:[UIColor blueColor] forState:UIControlStateNormal];
        [_downloadButton.titleLabel setFont:[UIFont systemFontOfSize:14]];
        [self.contentView addSubview:_downloadButton];
        [_downloadButton sizeToFit];
        _downloadButton.layer.cornerRadius = _downloadButton.frame.size.height * 0.5;
        _downloadButton.layer.borderColor = [UIColor blueColor].CGColor;
        _downloadButton.layer.borderWidth = 0.5;
        [_downloadButton addTarget:self action:@selector(downloadButtonClicked) forControlEvents:UIControlEventTouchUpInside];
    }
    
    return _downloadButton;
}

- (UIButton *)cancelButton {
    if (_cancelButton == nil) {
        _cancelButton = [UIButton buttonWithType:UIButtonTypeSystem];
        [_cancelButton setTitle:@"取消下载" forState:UIControlStateNormal];
        [_cancelButton setTitleColor:[UIColor blueColor] forState:UIControlStateNormal];
        [_cancelButton.titleLabel setFont:[UIFont systemFontOfSize:14]];
        [self.contentView addSubview:_cancelButton];
        [_cancelButton sizeToFit];
        _cancelButton.layer.cornerRadius = _cancelButton.frame.size.height * 0.5;
        _cancelButton.layer.borderColor = [UIColor blueColor].CGColor;
        _cancelButton.layer.borderWidth = 0.5;
        [_cancelButton addTarget:self action:@selector(cancelButtonClicked) forControlEvents:UIControlEventTouchUpInside];
    }
    
    return _cancelButton;
}

- (UIButton *)deleteButton {
    if (_deleteButton == nil) {
        _deleteButton = [UIButton buttonWithType:UIButtonTypeSystem];
        [_deleteButton setTitle:@"删除" forState:UIControlStateNormal];
        [_deleteButton setTitleColor:[UIColor redColor] forState:UIControlStateNormal];
        [_deleteButton.titleLabel setFont:[UIFont systemFontOfSize:14]];
        [self.contentView addSubview:_deleteButton];
        [_deleteButton sizeToFit];
        _deleteButton.layer.cornerRadius = _deleteButton.frame.size.height * 0.5;
        _deleteButton.layer.borderColor = [UIColor redColor].CGColor;
        _deleteButton.layer.borderWidth = 0.5;
        [_deleteButton addTarget:self action:@selector(deleteButtonClicked) forControlEvents:UIControlEventTouchUpInside];
    }
    
    return _deleteButton;
}

- (instancetype)initWithStyle:(UITableViewCellStyle)style reuseIdentifier:(NSString *)reuseIdentifier {
    if (self = [super initWithStyle:style reuseIdentifier:reuseIdentifier]) {
        [self setupSubviews];
    }
    
    return self;
}

- (void)setupSubviews {
    _titleLabel = [[UILabel alloc] init];
    _titleLabel.font = [UIFont systemFontOfSize:16];
    _titleLabel.textColor = [UIColor blackColor];
    [self.contentView addSubview:_titleLabel];
}

- (void)setItem:(QOfflineItem *)item {
    _item = item;
    
    _titleLabel.text = _item.name;
    
    if (_item.size > 0) {
        if (_sizeLabel == nil) {
            _sizeLabel = [[UILabel alloc] init];
            _sizeLabel.font = [UIFont systemFontOfSize:14];
            _sizeLabel.textColor = [UIColor grayColor];
            [self.contentView addSubview:_sizeLabel];
        }
        _sizeLabel.text = [NSString stringWithFormat:@"(%lld M)", _item.size / 1024 / 1024];
    }
    
    if (item.size > 0) {
        if ((item.status == QOfflineItemStatusNone || item.status == QOfflineItemStatusOutdated)) {
            // 下载
            self.downloadButton.hidden = NO;
            self.deleteButton.hidden = YES;
            self.cancelButton.hidden = YES;
        } else {
            // 已下载
            self.deleteButton.hidden = NO;
            self.downloadButton.hidden = YES;
            self.cancelButton.hidden = YES;
        }
    }
    
    // 判断当前是否正在下载
    for (QOfflineItem *downloadingItem in [QOfflineMap sharedOfflineMap].downloadingItems) {
        if (_item == downloadingItem) {
            self.deleteButton.hidden = YES;
            self.downloadButton.hidden = YES;
            self.cancelButton.hidden = NO;
        }
    }
}

- (void)downloadButtonClicked {
    if ([self.delegate respondsToSelector:@selector(offlineDownloadCell:downloadWithItem:)]) {
        [self.delegate offlineDownloadCell:self downloadWithItem:_item];
    }
}

- (void)cancelButtonClicked {
    if ([self.delegate respondsToSelector:@selector(offlineDownloadCell:cancelDownloadWithItem:)]) {
        [self.delegate offlineDownloadCell:self cancelDownloadWithItem:_item];
    }
}

- (void)deleteButtonClicked {
    if ([self.delegate respondsToSelector:@selector(offlineDownloadCell:deleteWithItem:)]) {
        [self.delegate offlineDownloadCell:self deleteWithItem:_item];
    }
}

- (void)layoutSubviews {
    [super layoutSubviews];
    
    [_titleLabel sizeToFit];
    _titleLabel.frame = CGRectMake(15, 0, _titleLabel.frame.size.width, _titleLabel.frame.size.height);
    _titleLabel.center = CGPointMake(_titleLabel.center.x, self.contentView.bounds.size.height * 0.5);
    
    if (_sizeLabel != nil) {
        [_sizeLabel sizeToFit];
        _sizeLabel.frame = CGRectMake(_titleLabel.frame.size.width + _titleLabel.frame.origin.x + 8, 0, _sizeLabel.frame.size.width, _sizeLabel.frame.size.height);
        _sizeLabel.center = CGPointMake(_sizeLabel.center.x, self.contentView.bounds.size.height * 0.5);
    }
    
    if (_downloadButton != nil) {
        [_downloadButton sizeToFit];
        CGFloat buttonWidth = _downloadButton.frame.size.width + 15;
        _downloadButton.frame = CGRectMake(self.contentView.frame.size.width - buttonWidth - 20, 0, buttonWidth, _downloadButton.frame.size.height);
        _downloadButton.center = CGPointMake(_downloadButton.center.x, self.contentView.bounds.size.height * 0.5);
    }
    
    if (_cancelButton != nil) {
        [_cancelButton sizeToFit];
        CGFloat buttonWidth = _cancelButton.frame.size.width + 15;
        _cancelButton.frame = CGRectMake(self.contentView.frame.size.width - buttonWidth - 20, 0, buttonWidth, _cancelButton.frame.size.height);
        _cancelButton.center = CGPointMake(_cancelButton.center.x, self.contentView.bounds.size.height * 0.5);
    }
    
    if (_deleteButton != nil) {
        [_deleteButton sizeToFit];
        CGFloat buttonWidth = _deleteButton.frame.size.width + 15;
        _deleteButton.frame = CGRectMake(self.contentView.frame.size.width - buttonWidth - 20, 0, buttonWidth, _deleteButton.frame.size.height);
        _deleteButton.center = CGPointMake(_deleteButton.center.x, self.contentView.bounds.size.height * 0.5);
    }
}

@end
