//
//  ClusterAnnotationViewController.m
//  QMapKitDemo
//
//  Created by Keith Cao on 2023/10/23.
//  Copyright © 2023 tencent. All rights reserved.
//

#import "ClusterAnnotationViewController.h"

@implementation TypedClustableAnnotation

- (id)copyWithZone:(nullable NSZone *)zone
{
	TypedClustableAnnotation* copy = [[super copyWithZone:zone] init];
	copy.type = self.type;
	return self;
}

@end


@interface ClusterAnnotationViewController () <QMUClusterProtocol>

@property (nonatomic, strong) NSMutableArray* originAnnotations;
@property (nonatomic, strong) QMUClusterManager*    manager;

@end

@implementation ClusterAnnotationViewController

- (void)viewDidLoad {
	[super viewDidLoad];
	
	self.mapView.zoomLevel = 8.830177;
	[self addClusters];
}

- (void)addClusters
{
	[self buildAnnotationsData];
	self.manager = [[QMUClusterManager alloc] initWithMap:self.mapView];
	self.manager.delegate = self;
	self.manager.distance = 16;
	self.manager.thresholdZoomlevel = 18;
	[self.manager addAnnotations:self.originAnnotations];
}


- (void)buildAnnotationsData
{
	self.originAnnotations = [NSMutableArray array];
	
	TypedClustableAnnotation *annotation = [[TypedClustableAnnotation alloc] init];
	annotation.coordinate = CLLocationCoordinate2DMake(39.984083,116.316515);
	annotation.type = 1;
	[self.originAnnotations addObject:annotation];
	
	for (int i = 0; i < 50; i++)
	{
		annotation = [[TypedClustableAnnotation alloc] init];
		annotation.coordinate = CLLocationCoordinate2DMake(39 + ((double)arc4random() / 0x100000000), 116 + ((double)arc4random() / 0x100000000));
		annotation.type = arc4random() % 3;
		[self.originAnnotations addObject:annotation];
	}
}

- (BOOL)clusterAnnotation:(QMUAnnotation*)anno1 withAnnotation:(QMUAnnotation*)anno2
{
	
	TypedClustableAnnotation *a = (TypedClustableAnnotation*)anno1;
	TypedClustableAnnotation *b = (TypedClustableAnnotation*)anno2;
	if (a.type == b.type)
	{
		return YES;
	}
	return NO;
}

#pragma mark -- Map Delegate

- (QAnnotationView *)mapView:(QMapView *)mapView viewForAnnotation:(id<QAnnotation>)annotation
{
	
	if ([annotation isKindOfClass:[QMUClusterAnnotation class]])
	{
		QMUClusterAnnotation *ca = (QMUClusterAnnotation*)annotation;
		
		// QMUClusterAnnotation 包含的标注点个数判断创建哪种类型的标注视图
		if (ca.count <= 1)
		{
			return [self buildNormalAnnotationView:annotation];
		}
		else
		{
			return [self buildClusterAnnotationView:ca];
		}
	}
	
	return nil;
}

- (QAnnotationView *)buildNormalAnnotationView:(QPointAnnotation *)annotation
{
	static NSString *pointReuseIndetifier = @"pointReuseIndetifier";
	QAnnotationView *annotationView = (QAnnotationView*)[self.mapView dequeueReusableAnnotationViewWithIdentifier:pointReuseIndetifier];
	
	if (annotationView == nil)
	{
		annotationView = [[QAnnotationView alloc] initWithAnnotation:annotation reuseIdentifier:pointReuseIndetifier];
	}
	
	annotationView.canShowCallout   = YES;
	annotationView.zIndex = 0;
	
	UIImage *img = [UIImage imageNamed:@"marker"];
	
	annotationView.image = img;
	annotationView.centerOffset = CGPointMake(0, -img.size.height / 2.0);
	
	return annotationView;
}

- (QMUClusterAnnotationView *)buildClusterAnnotationView:(QMUClusterAnnotation *)annotation
{
	static NSString *clusterReuseIndetifier = @"cluster";
	QMUClusterAnnotationView *annotationView = (QMUClusterAnnotationView*)[self.mapView dequeueReusableAnnotationViewWithIdentifier:clusterReuseIndetifier];
	if (annotationView == nil)
	{
		annotationView = [[QMUClusterAnnotationView alloc] initWithAnnotation:annotation
															  reuseIdentifier:clusterReuseIndetifier];
	}
	annotationView.backgroundColor = [UIColor colorWithRed:0.5 green:0.6 blue:0.45 alpha:0.8];
	annotationView.displayText.text = [[NSNumber numberWithUnsignedInteger:annotation.count] stringValue];
	
	if (annotation.count >= 2) {
		annotationView.displayText.text = [[[NSNumber numberWithUnsignedInteger:annotation.count] stringValue] stringByAppendingString:@"+"];
		annotationView.backgroundColor = [UIColor colorWithRed:0.9 green:0.5 blue:0.2 alpha:0.7];
	}
	annotationView.canShowCallout               = YES;
	annotationView.draggable                    = NO;
	annotationView.bounds = CGRectMake(0, 0, 2*self.manager.distance, 2*self.manager.distance);
	annotationView.layer.cornerRadius = self.manager.distance;
	annotationView.zIndex = (int)annotation.count;
	
	return annotationView;
}

- (void)mapView:(QMapView *)mapView regionDidChangeAnimated:(BOOL)animated gesture:(BOOL)bGesture
{
	[self.manager refreshCluster];
}

@end
