//
//  QMapServices.h
//  QMapKit
//
//  Copyright (c) 2017年 Tencent. All rights reserved.
//

#import <Foundation/Foundation.h>

/**
 *	@brief 地图缓存清理类型
 */
typedef NS_ENUM(NSInteger, QMapCacheClearType){
	QMapCacheClearTypeCityData = 1	///< 清理城市缓存
};

/**
 *	@brief 缓存清理选项类
 */
@interface QMapCacheOption : NSObject

/**
 *	@brief 缓存清理类型，默认QMapCacheClearTypeCityData
 */
@property (nonatomic, assign) QMapCacheClearType type;

/**
 *	@brief 缓存清理的阈值, 缓存大小大于此值时会对指定的缓存进行清理，单位：MB。[1, size]，默认值：1。
 */
@property (nonatomic, assign) int sizeLimit;

@end

/**
 * @brief 地图服务类
 */
@interface QMapServices : NSObject

/**
 * @brief 获取单例
 */
+ (nonnull QMapServices *)sharedServices;

/**
 * @brief apikey 在官网申请的服务授权标识的key
 */
@property (nonnull, copy, nonatomic) NSString *APIKey;

/**
 * @brief 自定义的用户ID，用于与开发者联调查询服务端日志的唯一标识
 */
@property (nullable, nonatomic, copy) NSString *userId;

/**
 * @brief 获取地图SDK的版本号
 */
- (NSString * _Nonnull)sdkVersion;

/**
 * @brief 隐私合规接口, 必须在地图初始化前调用, 默认为false
 *        设置为false, 地图实例返回nil, 地图功能不能使用
 *        隐私政策官网: https://lbs.qq.com/userAgreements/agreements/privacy
 */
- (void)setPrivacyAgreement:(BOOL)isAgree;

/**
*  @brief  是否开启debug 日志. 默认为NO.
*  @important: 正式发布版，切记不要开启.
*/
@property (nonatomic) BOOL enableDebugLog;

/**
 *  @brief  地图缓存配置项
 *  @param	cacheOptions 缓存清理选项类数组
 * 	@note	1. 配置后，每一个地图实例创建时会进行标注清理，仅在初始化地图时生效一次
 */
- (void)ConfigMapCacheClearOption:(NSArray <QMapCacheOption *> * _Nullable)cacheOptions;

/**
 *  @brief  获取地图缓存清理上限大小
 *  @param	type 地图缓存清理类型
 * 	@note	大小 单位MB，如未设置则返回-1
 */
- (int)getMapCacheClearSizeLimitByType:(QMapCacheClearType)type;

@end
