//
//  BusingSearchViewController.m
//  QMapKitDemo
//
//  Created by Keith Cao on 2023/10/18.
//  Copyright © 2023 tencent. All rights reserved.
//

#import "BusingSearchViewController.h"
#import <QMapKit/QMSSearchKit.h>
#import "QSearchTools.h"

@interface BusingSearchViewController () <QMSSearchDelegate>

@property (nonatomic, strong) QMSSearcher *mySearcher;

@end

@implementation BusingSearchViewController

- (void)searchBusRoute
{
	QMSBusingRouteSearchOption *busOpt = [[QMSBusingRouteSearchOption alloc] init];
	[busOpt setFrom:@"39.954355,116.416614"];
	[busOpt setTo:@"39.867821,116.495544"];
	[busOpt setPolicyWithType:QMSBusingRoutePolicyTypeLeastTime];
	
	QMapPoint points[2] = {QMapPointForCoordinate(CLLocationCoordinate2DMake(39.954355,116.416614)),
							QMapPointForCoordinate(CLLocationCoordinate2DMake(39.867821,116.495544))};
	
	QMapRect rect = QBoundingMapRectWithPoints(points, 2);
	
	// 调整视野
	[self.mapView setVisibleMapRect:rect edgePadding:UIEdgeInsetsMake(50, 30, 30, 50) animated:YES];
	
	[self.mySearcher searchWithBusingRouteSearchOption:busOpt];
}

- (void)drawRouteLineWithResult:(QMSBusingRouteSearchResult *)result
{
	[self.mapView removeOverlays:self.mapView.overlays];
	
	QMSBusingRoutePlan *plan = result.routes.firstObject;
	
	// 公交规划路线分段信息
	NSArray <QMSBusingSegmentRoutePlan *> *steps = plan.steps;
	
	for (QMSBusingSegmentRoutePlan *segRoutePlan in steps)
	{
		
		QPolyline *polyline = nil;
		// 路径规划类型 "DRIVING":驾车 "WALKING":步行 "TRANSIT":公交
		if ([segRoutePlan.mode isEqualToString:@"WALKING"])
		{
			polyline = [self createPolylineWithResultPoints:segRoutePlan.polyline];
		}
		// 非步行使用 QMSBusingRouteTransitLine 的点绘制路线
		else
		{
			for (QMSBusingRouteTransitLine *line in segRoutePlan.lines)
			{
				polyline = [self createPolylineWithResultPoints:line.polyline];
			}
		}
		
		polyline.userData = segRoutePlan.mode;
		[self.mapView addOverlay:polyline];
	}
}

// 经纬度转换
- (QPolyline *)createPolylineWithResultPoints:(NSArray <NSValue *> *)points
{
	NSUInteger count = points.count;
	if (count <2) return nil;
	CLLocationCoordinate2D coordinateArray[count];
	for (int i = 0; i < count; ++i)
	{
		coordinateArray[i] = [QSearchTools decodeCoordinateValue:[points objectAtIndex:i]];
	}
	QPolyline *polyline = [[QPolyline alloc] initWithCoordinates:coordinateArray count:count];
	return polyline;
}

- (QOverlayView *)mapView:(QMapView *)mapView viewForOverlay:(id<QOverlay>)overlay
{
	if ([overlay isKindOfClass:[QPolyline class]])
	{
		QPolyline *polyline = (QPolyline*)overlay;
		QTexturePolylineView *polylineRender = [[QTexturePolylineView alloc] initWithPolyline:polyline];
		NSString *mode = (NSString *)polyline.userData;
		if ([mode isEqualToString:@"WALKING"])
		{
			polylineRender.drawType = QTextureLineDrawType_FootPrint;
			polylineRender.styleTextureImage = [UIImage imageNamed:@"foot.jpg"];
			QSegmentStyle *segStyle = [[QSegmentStyle alloc] init];
			segStyle.startIndex = 0;
			segStyle.endIndex =(int)polyline.pointCount - 1;
			segStyle.colorImageIndex = 1;
			polylineRender.segmentStyle = @[segStyle];
		}
		else
		{
			polylineRender.drawType = QTextureLineDrawType_ColorLine;
			polylineRender.strokeColor = [UIColor greenColor];
			QSegmentColor *segColor = [[QSegmentColor alloc] init];
			segColor.startIndex = 0;
			segColor.endIndex =(int)polyline.pointCount - 1;
			segColor.color = [UIColor greenColor];
			polylineRender.segmentColor = @[segColor];
		}
		
		
		polylineRender.lineWidth = 8;
		return polylineRender;
	}
	return nil;
}

#pragma mark - SearchDelegate

- (void)searchWithSearchOption:(QMSSearchOption *)searchOption didFailWithError:(NSError *)error
{
	NSLog(@"error: %@", error);
}
- (void)searchWithBusingRouteSearchOption:(QMSBusingRouteSearchOption *)busingRouteSearchOption didRecevieResult:(QMSBusingRouteSearchResult *)busingRouteSearchResult
{
	[self drawRouteLineWithResult:busingRouteSearchResult];
	
	NSLog(@"bus result is: %@", busingRouteSearchResult);
}

- (void)viewDidLoad {
	[super viewDidLoad];
	
	self.mySearcher = [[QMSSearcher alloc] initWithDelegate: self];
	
	[self searchBusRoute];
}

@end
